(function () {
    const materialSelect      = document.getElementById("material");
    const pricePerKgInput     = document.getElementById("pricePerKg");
    const modelWeightInput    = document.getElementById("modelWeight");
    const supportWeightInput  = document.getElementById("supportWeight");
    const printTimeInput      = document.getElementById("printTime");
    const powerInput          = document.getElementById("power");
    const kwhPriceInput       = document.getElementById("kwhPrice");

    const wearPerHourInput    = document.getElementById("wearPerHour");
    const laborRateInput      = document.getElementById("laborRate");
    const laborHoursInput     = document.getElementById("laborHours");

    const markupPercentInput  = document.getElementById("markupPercent");
    const taxRateInput        = document.getElementById("taxRate");

    const customerNameInput   = document.getElementById("customerName");
    const offerNumberInput    = document.getElementById("offerNumber");
    const offerNotesInput     = document.getElementById("offerNotes");
    const shippingNetInput   = document.getElementById("shippingNet");

    const totalCostEl    = document.getElementById("totalCost");
    const materialCostEl = document.getElementById("materialCost");
    const energyCostEl   = document.getElementById("energyCost");
    const wearCostEl     = document.getElementById("wearCost");
    const laborCostEl    = document.getElementById("laborCost");
    const perHourEl      = document.getElementById("perHour");
    const perGramEl      = document.getElementById("perGram");
    const totalWeightEl  = document.getElementById("totalWeight");
    const salePriceNetEl = document.getElementById("salePriceNet");
    const vatAmountEl    = document.getElementById("vatAmount");
    const salePriceGrossEl = document.getElementById("salePriceGross");
    const profitAbsEl    = document.getElementById("profitAbs");
    const warningEl      = document.getElementById("warning");

    const calcBtn        = document.getElementById("calcBtn");
    const pdfBtn         = document.getElementById("pdfBtn");

    let lastCalc = null;

    function setPriceFromMaterial() {
      const selectedOption = materialSelect.options[materialSelect.selectedIndex];
      const price = parseFloat(selectedOption.dataset.price || "0");
      // Immer Materialpreis überschreiben, wenn Material gewechselt wird
      pricePerKgInput.value = isFinite(price) ? price.toFixed(2) : "";
    }

    materialSelect.addEventListener("change", setPriceFromMaterial);
    setPriceFromMaterial();

    function formatEuro(value) {
      if (!isFinite(value)) return "–";
      return value.toFixed(2).replace(".", ",") + " €";
    }

    function formatNumber(value, decimals = 2, suffix = "") {
      if (!isFinite(value)) return "–";
      return value.toFixed(decimals).replace(".", ",") + (suffix || "");
    }

    function parseOrZero(input) {
      if (!input || typeof input.value !== "string") return 0;
      const v = parseFloat(input.value.replace(",", "."));
      return isFinite(v) ? v : 0;
    }

    function calculate() {
      warningEl.style.display = "none";

      const pricePerKg   = parseFloat((pricePerKgInput.value || "").replace(",", "."));
      const modelWeight  = parseFloat((modelWeightInput.value || "").replace(",", "."));
      const supportWeight= parseFloat((supportWeightInput.value || "0").replace(",", "."));
      const printTime    = parseFloat((printTimeInput.value || "").replace(",", "."));
      const power        = parseFloat((powerInput.value || "").replace(",", "."));
      const kwhPrice     = parseFloat((kwhPriceInput.value || "").replace(",", "."));

      const wearPerHour  = parseOrZero(wearPerHourInput);
      const laborRate    = parseOrZero(laborRateInput);
      const laborHours   = parseOrZero(laborHoursInput);
      const markup       = parseOrZero(markupPercentInput);
      const taxRate      = parseOrZero(taxRateInput);

      const requiredFields = [pricePerKg, modelWeight, printTime, power, kwhPrice];
      if (requiredFields.some(v => !isFinite(v) || v < 0)) {
        warningEl.style.display = "block";
        totalCostEl.textContent     = "–";
        materialCostEl.textContent  = "–";
        energyCostEl.textContent    = "–";
        wearCostEl.textContent      = "–";
        laborCostEl.textContent     = "–";
        perHourEl.textContent       = "–";
        perGramEl.textContent       = "–";
        totalWeightEl.textContent   = "–";
        salePriceNetEl.textContent  = "–";
        vatAmountEl.textContent     = "–";
        salePriceGrossEl.textContent= "–";
        profitAbsEl.textContent     = "–";
        lastCalc = null;
        return;
      }

      const totalWeightG  = modelWeight + (isFinite(supportWeight) ? supportWeight : 0);
      const totalWeightKg = totalWeightG / 1000;

      const materialCost  = totalWeightKg * pricePerKg;
      const powerKw       = power / 1000;
      const energyCost    = printTime * powerKw * kwhPrice;

      const wearCost      = printTime * wearPerHour;
      const laborCost     = laborHours * laborRate;

      const totalCost     = materialCost + energyCost + wearCost + laborCost;
      const perHour       = printTime > 0 ? totalCost / printTime : NaN;
      const perGram       = totalWeightG > 0 ? totalCost / totalWeightG : NaN;

      const saleNet       = totalCost * (1 + (markup / 100));
      const vatAmount     = saleNet * (taxRate / 100);
      const saleGross     = saleNet + vatAmount;
      const profitAbs     = saleNet - totalCost;

      totalCostEl.textContent     = formatEuro(totalCost);
      materialCostEl.textContent  = formatEuro(materialCost);
      energyCostEl.textContent    = formatEuro(energyCost);
      wearCostEl.textContent      = formatEuro(wearCost);
      laborCostEl.textContent     = formatEuro(laborCost);
      perHourEl.textContent       = formatEuro(perHour);
      perGramEl.textContent       = formatNumber(perGram, 4, " €/g");
      totalWeightEl.textContent   = formatNumber(totalWeightG, 1, " g");
      salePriceNetEl.textContent  = formatEuro(saleNet);
      vatAmountEl.textContent     = formatEuro(vatAmount);
      salePriceGrossEl.textContent= formatEuro(saleGross);
      profitAbsEl.textContent     = formatEuro(profitAbs);

      lastCalc = {
        totalCost: totalCost,
        materialCost: materialCost,
        energyCost: energyCost,
        wearCost: wearCost,
        laborCost: laborCost,
        perHour: perHour,
        perGram: perGram,
        totalWeightG: totalWeightG,
        saleNet: saleNet,
        vatAmount: vatAmount,
        saleGross: saleGross,
        profitAbs: profitAbs,
        markup: markup,
        taxRate: taxRate
      };
    }

    function escapeHtml(str) {
      return String(str || "").replace(/[&<>]/g, function (c) {
        return ({ "&": "&amp;", "<": "&lt;", ">": "&gt;" })[c] || c;
      });
    }

    function openOfferPdf() {
      calculate();
      if (!lastCalc) {
        alert("Bitte zuerst eine gültige Kalkulation durchführen.");
        return;
      }

      const customer = escapeHtml(customerNameInput.value || "Kunde");
      const offerNo  = escapeHtml(offerNumberInput.value || "–");
      const notes    = escapeHtml(offerNotesInput.value || "");
      const today    = new Date().toLocaleDateString("de-DE");

      const shippingNetValue = (shippingNetInput.value || "").replace(",", ".");
      const shippingNet = parseFloat(shippingNetValue) || 0;
      const shippingVat = shippingNet * (lastCalc.taxRate / 100);
      const shippingGross = shippingNet + shippingVat;

      const invoiceNet = lastCalc.saleNet + shippingNet;
      const invoiceVat = lastCalc.vatAmount + shippingVat;
      const invoiceGross = lastCalc.saleGross + shippingGross;

      const shippingRow = shippingNet > 0 ? `
      <tr>
        <td>2</td>
        <td>Versand innerhalb Deutschlands</td>
        <td class="right">1,00 Stk</td>
        <td class="right">${shippingNet.toFixed(2).replace(".", ",")} €</td>
        <td class="right">${shippingNet.toFixed(2).replace(".", ",")} €</td>
      </tr>` : "";

      const html = `<!DOCTYPE html>
<html lang="de">
<head>
  <meta charset="UTF-8" />
  <title>Angebot 3D-Druck</title>
  <style>
    body {
      font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif;
      margin: 24px;
      color: #111827;
    }
    h1 {
      font-size: 1.4rem;
      margin-bottom: 4px;
    }
    .meta {
      font-size: 0.85rem;
      margin-bottom: 16px;
    }
    .meta div {
      margin-bottom: 2px;
    }
    table {
      width: 100%;
      border-collapse: collapse;
      margin-top: 12px;
      margin-bottom: 16px;
      font-size: 0.9rem;
    }
    th, td {
      border: 1px solid #d1d5db;
      padding: 6px 8px;
      text-align: left;
    }
    th {
      background: #f3f4f6;
    }
    tfoot td {
      font-weight: 600;
    }
    .right { text-align: right; }
    .small {
      font-size: 0.8rem;
      color: #4b5563;
    }
    .notes {
      margin-top: 10px;
      font-size: 0.85rem;
    }
  </style>
</head>
<body>
  <h1>Angebot 3D-Druck</h1>
  <div class="meta">
    <div><strong>Kunde / Projekt:</strong> ${customer}</div>
    <div><strong>Angebotsnummer:</strong> ${offerNo}</div>
    <div><strong>Datum:</strong> ${today}</div>
  </div>

  <table>
    <thead>
      <tr>
        <th>Position</th>
        <th>Beschreibung</th>
        <th class="right">Menge</th>
        <th class="right">Einzelpreis (netto)</th>
        <th class="right">Gesamt (netto)</th>
      </tr>
    </thead>
    <tbody>
      <tr>
        <td>1</td>
        <td>3D-Druckdienstleistung inkl. Material<br>
            <span class="small">
              Material, Strom, Verschleiß, Personalkosten<br>
              Kalkulationsaufschlag: ${lastCalc.markup.toFixed(0).replace(".", ",")}%
            </span>
        </td>
        <td class="right">1,00 Stk</td>
        <td class="right">${lastCalc.saleNet.toFixed(2).replace(".", ",")} €</td>
        <td class="right">${lastCalc.saleNet.toFixed(2).replace(".", ",")} €</td>
      </tr>
      ${shippingRow}
    </tbody>
    <tfoot>
      <tr>
        <td colspan="4" class="right">Zwischensumme (netto)</td>
        <td class="right">${invoiceNet.toFixed(2).replace(".", ",")} €</td>
      </tr>
      <tr>
        <td colspan="4" class="right">zzgl. MwSt. (${lastCalc.taxRate.toFixed(1).replace(".", ",")} %)</td>
        <td class="right">${invoiceVat.toFixed(2).replace(".", ",")} €</td>
      </tr>
      <tr>
        <td colspan="4" class="right">Gesamtbetrag (brutto)</td>
        <td class="right">${invoiceGross.toFixed(2).replace(".", ",")} €</td>
      </tr>
    </tfoot>
  </table>

  <div class="notes">
    <strong>Hinweise:</strong><br>
    ${notes || "<span class='small'>Zahlungskonditionen, Lieferzeit o.Ä. hier eintragen.</span>"}
  </div>

  <p class="small" style="margin-top:16px;">
    Dieses Angebot basiert auf den zum Zeitpunkt der Erstellung kalkulierten Material-, Energie- und Personalkosten
    und versteht sich, sofern nicht anders angegeben, als freibleibend.
  </p>
</body>
</html>`;

      const w = window.open("", "_blank");
      if (!w) {
        alert("Popup-Fenster wurde vom Browser blockiert. Bitte Popups erlauben und erneut versuchen.");
        return;
      }
      w.document.open();
      w.document.write(html);
      w.document.close();
      w.focus();
    }

    calcBtn.addEventListener("click", calculate);

    [
      pricePerKgInput, modelWeightInput, supportWeightInput,
      printTimeInput, powerInput, kwhPriceInput,
      wearPerHourInput, laborRateInput, laborHoursInput,
      markupPercentInput, taxRateInput
    ].forEach(input => {
      input.addEventListener("input", calculate);
    });

    pdfBtn.addEventListener("click", openOfferPdf);
  })();